<?php

require_once('SSHCommand.php');
require_once('DatabaseConfigManager.php');

function save_cluster_esxi_conf($argv)
{
    global $DTASCFG_XML_FILE;
    $ret = parseClusterESXiInfo($argv, $sc_id, $esxi_info);
    if (!$ret) {
        echo "Usage: save_cluster_esxi_conf SC_ID\n";
        exit(1);
    }

    try {
        save_cluster_esxi_conf_to_master($DTASCFG_XML_FILE, $sc_id, $esxi_info);
        save_cluster_esxi_conf_to_slave($DTASCFG_XML_FILE, $sc_id, $esxi_info, '/etc/dtascfg.xml.backup');
        save_cluster_esxi_conf_to_slave_usbx($DTASCFG_XML_FILE, $sc_id, $esxi_info);
    } catch (Exception $e) {
        debug_print("Configure ESXi setting fail," . var_export($e, true));
        echo "Configure ESXi setting fail " . $e->getMessage();
        exit(1);
    }
    exit(0);
}

function parseClusterESXiInfo($argv, &$sc_id, &$esxi_info)
{
    if (count($argv) != 4) {
        return false;
    }
    $sc_id = (int)reset($argv);
    $esxi_host = next($argv);
    $esxi_user = next($argv);
    $esxi_pwd = next($argv);
    $esxi_info = new ESXiInfo($esxi_host, $esxi_user, $esxi_pwd);
    return true;
}

function save_cluster_esxi_conf_to_master($cfg_xml_path, $sc_id, $esxi_info)
{
    $cluster_esxi_configuration = new MasterESXiConfiguration($cfg_xml_path, $sc_id, $esxi_info);
    $cluster_esxi_configuration->set2ManagementServerDB();
}

function save_cluster_esxi_conf_to_slave($cfg_xml_path, $sc_id, $esxi_info, $backup_cfg_path)
{
    $cluster_esxi_configuration = new SlaveESXiConfiguration($cfg_xml_path, $sc_id, $esxi_info, $backup_cfg_path);
    $cluster_esxi_configuration->set2SlaveManagementServerDB();
    $cluster_esxi_configuration->set2SlaveSandboxControllerBackupConfig();
}

function save_cluster_esxi_conf_to_slave_usbx($cfg_xml_path, $sc_id, $esxi_info)
{
    $usbx_esxi_setting_retriever = new UsbxESXiSettingRetriever($cfg_xml_path, $sc_id);
    $esxi_setting = $usbx_esxi_setting_retriever->getESXiSetting();
    $esxi_setting->setESXi($esxi_info);
}

class MasterESXiConfiguration
{
    protected $_sc_id = null;
    protected $_ms_db_cfg_mgr = null;
    protected $_esxi_info = null;

    function __construct($cfg_xml_path, $sc_id, $esxi_info)
    {
        $this->_cfg_xml_path = $cfg_xml_path;
        $this->_sc_id = $sc_id;
        $this->_esxi_info = $esxi_info;
        $this->_ms_db_cfg_mgr = $this->initDBConfigManager();
    }

    protected function initDBConfigManager()
    {
        $ms_ssh_cmd_retriever = new ManagementServerSSHCmdRetriever($this->_cfg_xml_path);
        $ms_ssh_cmd = $ms_ssh_cmd_retriever->getSSHCmd();
        $ms_db_info_retriever = new ManagementServerDBInfoRetriever($this->_cfg_xml_path);
        $ms_db_info = $ms_db_info_retriever->getDBInfo();
        return new DatabaseConfigManager($ms_db_info, $ms_ssh_cmd);
    }

    public function set2ManagementServerDB()
    {
        $this->_ms_db_cfg_mgr->setClusterConfig(DBKeyConstant::DB_KEY_MS_IP, $this->_esxi_info->getHost(), $this->_sc_id);
        $this->_ms_db_cfg_mgr->setClusterConfig(DBKeyConstant::DB_KEY_SC_IP, $this->_esxi_info->getHost(), $this->_sc_id);
        $this->_ms_db_cfg_mgr->setClusterConfig(DBKeyConstant::DB_KEY_ESXI_IP, $this->_esxi_info->getHost(), $this->_sc_id);
        $this->_ms_db_cfg_mgr->setClusterConfig(DBKeyConstant::DB_KEY_ESXI_USER, $this->_esxi_info->getUser(), $this->_sc_id);
        $this->_ms_db_cfg_mgr->setClusterConfig(DBKeyConstant::DB_KEY_ESXI_PWD, $this->_esxi_info->getEscapePwd(), $this->_sc_id);
        $this->_ms_db_cfg_mgr->notifyConfigChange();
    }
}

class DBKeyConstant
{
    const DB_KEY_MS_IP = "configuration.management_server.ip";
    const DB_KEY_SC_IP = "configuration.sandbox_controller.ip";
    const DB_KEY_ESXI_IP = "configuration.esxi.ip";
    const DB_KEY_ESXI_USER = "configuration.esxi.username";
    const DB_KEY_ESXI_PWD = "configuration.esxi.password";
    const XPATH_RELATIVE_MS_IP = 'management_server/ip';
    const XPATH_RELATIVE_ESXI_PWD = 'esxi/password';
    const XPATH_RELATIVE_ESXI_USER = 'esxi/username';
}

class SlaveESXiConfiguration
{
    protected $_esxi_info = null;
    protected $_slave_ms_db_cfg_mgr = null;
    protected $_slave_sc_cfg_setting = null;
    protected $_sc_id = null;
    protected $_cfg_xml_path = null;
    protected $__backup_cfg_path = null;

    function __construct($cfg_xml_path, $sc_id, $esxi_info, $backup_cfg_path)
    {
        $this->_cfg_xml_path = $cfg_xml_path;
        $this->_backup_cfg_path = $backup_cfg_path;
        $this->_sc_id = $sc_id;
        $this->_esxi_info = $esxi_info;

        $this->_slave_ms_db_cfg_mgr = $this->initSlaveDBConfigManager();
        $this->_slave_sc_cfg_setting = $this->initSlaveSandboxControllerConfigSetting();
    }

    protected function initSlaveSandboxControllerConfigSetting()
    {
        $cluster_sc_ssh_cmd_retriever = new ClusterSandboxControllerSSHCmdRetriever($this->_cfg_xml_path, $this->_sc_id);
        $cluster_sc_ssh_cmd = $cluster_sc_ssh_cmd_retriever->getSSHCmd();
        return new ConfigSetting($this->_backup_cfg_path, $cluster_sc_ssh_cmd);
    }

    protected function initSlaveDBConfigManager()
    {
        $cluster_ms_ssh_cmd_retriever = new ClusterManagementServerSSHCmdRetriever($this->_cfg_xml_path, $this->_sc_id);
        $cluster_ms_ssh_cmd = $cluster_ms_ssh_cmd_retriever->getSSHCmd();
        $cluster_ms_db_info_retriever = new ClusterManagementServerDBInfoRetriever($this->_cfg_xml_path, $this->_sc_id);
        $db_info = $cluster_ms_db_info_retriever->getDBInfo();
        return new DatabaseConfigManager($db_info, $cluster_ms_ssh_cmd);
    }

    public function set2SlaveManagementServerDB()
    {
        $this->_slave_ms_db_cfg_mgr->setConfig(DBKeyConstant::DB_KEY_MS_IP, $this->_esxi_info->getHost());
        $this->_slave_ms_db_cfg_mgr->setConfig(DBKeyConstant::DB_KEY_ESXI_USER, $this->_esxi_info->getUser());
        $this->_slave_ms_db_cfg_mgr->setConfig(DBKeyConstant::DB_KEY_ESXI_PWD, $this->_esxi_info->getEscapePwd());
        $this->_slave_ms_db_cfg_mgr->notifyConfigChange();
    }

    public function set2SlaveSandboxControllerBackupConfig()
    {
        $this->_slave_sc_cfg_setting->setConfig(DBKeyConstant::XPATH_RELATIVE_MS_IP, $this->_esxi_info->getHost());
        $this->_slave_sc_cfg_setting->setConfig(DBKeyConstant::XPATH_RELATIVE_ESXI_USER, $this->_esxi_info->getUser());
        $this->_slave_sc_cfg_setting->setConfig(DBKeyConstant::XPATH_RELATIVE_ESXI_PWD, $this->_esxi_info->getEscapePwd());
    }
}

class ESXiInfo
{
    protected $_host = null;
    protected $_user = null;
    protected $_pwd = null;

    function __construct($host, $user, $pwd)
    {
        $this->setHost($host);
        $this->setUser($user);
        $this->setPwd($pwd);
    }

    public function setHost($host)
    {
        $this->_host = $host;
    }

    public function getHost()
    {
        return $this->_host;
    }

    public function setPwd($pwd)
    {
        $this->_pwd = $pwd;
    }

    public function getPwd()
    {
        return $this->_pwd;
    }

    public function getEscapePwd()
    {
        return escapeshellarg($this->_pwd);
    }

    public function setUser($user)
    {
        $this->_user = $user;
    }

    public function getUser()
    {
        return $this->_user;
    }
}


Class UsbxESXiSettingRetriever
{
    protected $_sc_id = null;
    protected $_cfg_xml = null;
    protected $_usbx_cli_path = null;
    protected $_ssh_cmd_retriever = null;

    function __construct($cfg_xml_path, $sc_id)
    {
        $this->_sc_id = $sc_id;
        $this->_cfg_xml = simplexml_load_file($cfg_xml_path);
        $this->_usbx_cli_path = (string)$this->_cfg_xml->u_sandbox->cli_path;
        $this->_ssh_cmd_retriever = new ClusterSandboxControllerSSHCmdRetriever($cfg_xml_path, $sc_id);
    }

    public function getESXiInfo()
    {
        $cluster_section = "cluster_$this->_sc_id";
        $ip = (string)$this->_cfg_xml->$cluster_section->esxi->ip;
        $pwd = (string)$this->_cfg_xml->$cluster_section->esxi->password;
        $user = (string)$this->_cfg_xml->$cluster_section->esxi->username;
        return new ESXiInfo($ip, $user, $pwd);
    }

    public function getESXiSetting()
    {
        $ssh_cmd = $this->_ssh_cmd_retriever->getSSHCmd();
        $esxi_setting = new UsbxESXiSetting($this->_usbx_cli_path, $ssh_cmd);
        return $esxi_setting;
    }

}

class ConfigSetting
{
    const CONFIG_SETTING_BIN_PATH = "/opt/TrendMicro/DTAS/SandboxController/bin/dtascfg";
    const DEF_XPATH_NODE_INDEX = 0;
    const XPATH_PATTERN = <<<heredoc
        /(?P<pre_xpath>.*)\/(?P<last_node_name>[^\/\[\]]+)(\[(?P<node_index>[0-9]+)\])?/
heredoc;
    const CONFIG_SET = 'set';
    protected $_ssh_cmd = null;
    protected $_def_cfg_seting_cmd_list = null;


    function __construct($cfg_xml_path, $ssh_cmd)
    {
        $this->_ssh_cmd = $ssh_cmd;
        $this->_def_cfg_seting_cmd_list = array(ConfigSetting::CONFIG_SETTING_BIN_PATH, $cfg_xml_path);
    }

    protected function getConfigSettingParamList($relative_xpath_to_root, $value)
    {
        $ret = preg_match(ConfigSetting::XPATH_PATTERN, $relative_xpath_to_root, $matches);
        if (!$ret) throw new  Exception("Regex match for config setting error: $relative_xpath_to_root");
        $pre_xpath = $matches['pre_xpath'];
        $last_node_name = $matches['last_node_name'];
        $node_index = $matches['node_index'] ? : ConfigSetting::DEF_XPATH_NODE_INDEX;
        return $cfg_setting_param_list = array($pre_xpath, $last_node_name, $value, $node_index);
    }

    public function setConfig($relative_xpath_to_root, $value)
    {
        $cfg_setting_param_list = $this->getConfigSettingParamList($relative_xpath_to_root, $value);
        $cfg_setting_cmd = array_merge($this->_def_cfg_seting_cmd_list, array(ConfigSetting::CONFIG_SET), $cfg_setting_param_list);
        $cfg_setting = implode(' ', $cfg_setting_cmd);
        $this->_ssh_cmd->doCmdAndCheckExitCode($cfg_setting);
    }
}

Class UsbxESXiSetting
{
    const ESXI_INTERNAL_IP = '169.254.3.1';
    protected $_ssh_cmd = null;
    protected $_def_esxi_seting_cmd_list = null;

    function __construct($usbx_cli_path, $ssh_cmd)
    {
        $this->_ssh_cmd = $ssh_cmd;
        $this->_def_esxi_seting_cmd_list = array($usbx_cli_path, 'set-esxi');
    }

    public function setESXi($esxi_info)
    {
        $set_esxi_parameter_list = array('--host', UsbxESXiSetting::ESXI_INTERNAL_IP, '--user', $esxi_info->getUser(), '--passwd', $esxi_info->getEscapePwd());
        $set_esxi_cmd_list = array_merge($this->_def_esxi_seting_cmd_list, $set_esxi_parameter_list);
        $set_esxi_cmd = implode(" ", $set_esxi_cmd_list);
        $out = $this->_ssh_cmd->doCmdAndCheckExitCode($set_esxi_cmd);
    }

}
